<?php

namespace App\Filament\Pages;

use App\Filament\Resources\SalesResource;
use Filament\Pages\Page;
use App\Models\Sales;
use Filament\Forms;
use Filament\Tables;
use Filament\Tables\Contracts\HasTable;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Table;
use Illuminate\Support\Carbon;
use Filament\Forms\Form;
use Filament\Tables\Filters\Filter;

class SalesReport extends Page implements HasTable
{
    use InteractsWithTable;

    protected static string $view = 'filament.pages.sales-report';
    protected static ?string $navigationIcon = 'heroicon-o-chart-bar';
    protected static ?string $title = 'Laporan Penjualan';

    public ?string $dateFrom = null;
    public ?string $dateTo = null;

    public function table(Table $table): Table
    {
        return $table
            ->query(
                Sales::query()
                    ->when($this->dateFrom, fn($q) => $q->where('sale_date', '>=', $this->dateFrom))
                    ->when($this->dateTo, fn($q) => $q->where('sale_date', '<=', $this->dateTo))
            )
            ->columns([
                Tables\Columns\TextColumn::make('invoice_number')->label('No. Nota'),
                Tables\Columns\TextColumn::make('sale_date')->label('Tanggal'),
                Tables\Columns\TextColumn::make('customer_name')->label('Customer'),
                Tables\Columns\TextColumn::make('payment_method')->label('Metode'),
                Tables\Columns\TextColumn::make('items_sum_subtotal')
                    ->label('Total')
                    ->money('IDR')
                    ->sortable()
                    ->searchable(),
            ])
            ->filters([
                Filter::make('sale_date')
                    ->form([
                        Forms\Components\DatePicker::make('dateFrom')->label('Dari'),
                        Forms\Components\DatePicker::make('dateTo')->label('Sampai'),
                    ])
                    ->query(function ($query, array $data) {
                        return $query
                            ->when($data['dateFrom'], fn ($q) => $q->where('sale_date', '>=', $data['dateFrom']))
                            ->when($data['dateTo'], fn ($q) => $q->where('sale_date', '<=', $data['dateTo']));
                    }),
            ])
            ->modifyQueryUsing(fn ($query) => $query->withSum('items as items_sum_subtotal', 'subtotal'));
    }

    public function getViewData(): array
    {
        $query = Sales::query()
            ->when($this->dateFrom, fn ($q) => $q->where('sale_date', '>=', $this->dateFrom))
            ->when($this->dateTo, fn ($q) => $q->where('sale_date', '<=', $this->dateTo))
            ->withSum('items as total', 'subtotal');

        $total = $query->get()->sum('total');

        return [
            'totalSales' => $total,
        ];
    }
}
